<?php
namespace WPSURL\App\Elementor;

use WPSURL\App\Assets;
use WPSURL\App\Traits\Singleton;

/**
 *
 * Block direct access to the main plugin file.
 *
 */
defined('ABSPATH') or die('No script kiddies please!');

final class Widgets {
    const VERSION = '1';
    const MINIMUM_ELEMENTOR_VERSION = '2.0.0';
    const MINIMUM_PHP_VERSION = '7.0';

    private static $widgets = [];
    private static $namespace = 'widget';

    use Singleton;

    /**
     * Elementor register widgets
     *
     * @return void
     */
    private static function registerWidgets(){
        /**
         *
         * Section Global widgets
         *
         */
        self::registerWidget('PageTitle');
        self::registerWidget('AlertBox');
        self::registerWidget('RedirectButton');
        self::registerWidget('ReportForm');
        self::registerWidget('Timer');
    }

    /**
     * Registered widgets append to elementor
     *
     * @return void
     */
    public static function widgetsRegistered($widgets_manager){
        self::registerWidgets();

        $widgets = self::getWidgets();

        if($widgets && is_array($widgets)){
            foreach($widgets as $widget){
                $class_name = $widget['namespace'] . '\\' . $widget['widget'];

                if(class_exists($class_name)){
                    $class = new $class_name();
                    $widgets_manager->register($class);
                }
            }
        }
    }

    /**
     * Elements namespaces
     *
     * @return string
     */
    private static function getNamespace(){
        $namespaces = array(
            'widget' => 'WPSURL\App\Elementor\Widgets',
        );

        return (!empty(self::$namespace) && isset($namespaces[self::$namespace])) ? $namespaces[self::$namespace] : self::$namespace;
    }

    /**
     * Set widget class namespace
     *
     * @param string $namespace
     * @return object
     */
    private static function namespace($namespace = 'widget'){
        self::$namespace = $namespace;
        return new self();
    }

    /**
     * Register a widget
     *
     * @param string $widget
     * @return void
     */
    private static function registerWidget($widget){
        self::$widgets[] = ['namespace' => self::getNamespace(), 'widget' => $widget];
    }

    /**
     * Get registered widgets
     *
     * @return array
     */
    public static function getWidgets(){
        return self::$widgets;
    }

    /**
     * Generate html notice minimum version
     *
     * @param string $property
     * @param string $minimum_version
     * @return string
     */
    public static function noticeMinimumVersion($property, $minimum_version, $current_version, $notice_type = 'warning'){
        if((!empty($minimum_version) && !empty($current_version)) && version_compare($current_version, $minimum_version, '<')){
            $str_before = '<strong>';
            $str_after = '</strong>';
            $message = sprintf(
                esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', WPSURL_TEXT_DOMAIN),
                $str_before . esc_html__(WPSURL_TEXT_DOMAIN, WPSURL_TEXT_DOMAIN) . $str_after,
                $str_before . esc_html__($property, WPSURL_TEXT_DOMAIN) . $str_after,
                $minimum_version
            );
            return sprintf('<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>', $notice_type, $message);
        }
        return false;
    }

    /**
     * Elementor minimum version check notice
     *
     * @return string
     */
    public static function checkRequirementsMinimumVersion(){
        echo self::noticeMinimumVersion('Elementor', self::MINIMUM_ELEMENTOR_VERSION, get_option('elementor_version'));
        echo self::noticeMinimumVersion('PHP', self::MINIMUM_PHP_VERSION, PHP_VERSION);
    }
    public static function registerWidgetsCategory($elements_manager){
        $elements_manager->add_category(
            'wpsurl_elements',
            [
                'title' => __('WP Safe Url Elements', WPSURL_TEXT_DOMAIN),
                'icon' => 'eicon-editor-link',
            ]
        );
    }

    public static function registerFontsGroup($font_groups) {
        $new_font_group = array( 'custom' => __( 'Custom' ) );
        return array_merge( $new_font_group, $font_groups );
    }

    public static function registerAdditionalFonts($additional_fonts) {
        $additional_fonts['IRSans'] = 'custom';
        return $additional_fonts;
    }

    public static function enqueueEditorStyles() {
        Assets::enqueueStyle('wpsurl.assets.css', WPSURL_CSS_URL . 'assets.css', null, WPSURL_VERSION);
    }
}